/** @file   highscoretable.h
 * @brief   Declaration of HighscoreTable - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_HIGHSCORETABLE_H
#define H_WWW_HIGHSCORETABLE_H

#include <string>
#include <vector>
#include <iostream>

namespace WeWantWar {


/** @class  HighscoreTable
 * @brief   Represents the highscore table.
 * @author  Tomi Lamminsaari
 *
 * Maintains the highscoretable.
 */
class HighscoreTable
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** The highscore entries are stored in a structures of this type.
   */
  struct Entry {
    /** Number of scores. */
    int         scores;
    /** Name of the player */
    std::string name;
  };
    


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor. Opens the highscore-file and reads the data.
   * @param     filename          Name of the highscore file.
   */
	HighscoreTable( const std::string& filename );


	/** Destructor
   */
	virtual ~HighscoreTable();

	/** Copy constructor.
   * @param     rO                Reference to another HighscoreTable
   */
  HighscoreTable( const HighscoreTable& rO );

	/** Assignment operator
   * @param     rO                Reference to another HighscoreTable
   * @return    Reference to us.
   */
  HighscoreTable& operator = ( const HighscoreTable& rO );




  ///
  /// Methods
  /// =======
  
  /** Suggests the given entry to this table. If scores are high enough, the
   * given data will be added to the table.
   * @param     scores            Number of scores
   * @param     rName             Name of the player
   */
  void suggestEntry( int scores, const std::string& rName );

  /** Saves the highscore table
   * @return    Nonzero if fails.
   */
  int save() const;



  ///
  /// Getter methods
  /// ==============
  
  /** Returns the position how high you got with given scores.
   * @param     score             Number of scores
   * @return    Position or -1 if scores are not high enough.
   */
  int getPosition( int score ) const;
  
  /** Returns the index'th entry
   * @param     index             The index of the entry being returned.
   * @return    The entry
   */
  Entry getEntry( int index ) const;
  
  /** Returns the number of entries.
   * @return    Number of entries the the highscorefile has.
   */
  int numberOfEntries() const;
  

protected:

 /** Creates new default highscoretable. This is called when we cannot
  * open the given highscorefile.
  */
  void createDefaultHighscores();
  
  /** Opens the highscorefile and reads the entries from there.
   */
  void openHighscoreFile();
  
  /** Reads the single highscore entry and pushed it to the @c m_entries
   * vector.
   * @param     rIn               The input stream
   * @return    Nonzero if errors
   */
  int readEntry( std::istream& rIn );
  
  

  ///
  /// Members
  /// =======
  
  /** Name of the highscorefile */
  std::string m_highscorefile;
  
  /** The highscore entries */
  std::vector<Entry> m_entries;
  



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace


#endif

/**
 * Version history
 * ===============
 * $Log: highscoretable.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:00+02  lamminsa
 * Initial revision
 *
 */
 
